/**
* \file: GstreamerVideoOut.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: Digital iPod Out
*
* \author: Veeraiyan Chidambaram /RBEI/ECF3/ veeraiyan.chidambaram@in.bosch.com
*          J. Harder / ADIT/SW1 / jharder@de.adit-jv.com
*
* \copyright (c) 2013 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include <memory.h>
#include <queue>
#include <gst/gst.h>
#include <gst/app/gstappsrc.h>
#include <uspi/EndpointDataSharing.h>
#include "GstreamerVideoOut.h"
#include "GstreamerCommon.h"

#include <inttypes.h>

using namespace std;

LOG_IMPORT_CONTEXT(dipo_gst);
#define GST_MAIN_LOOP_PRIO_DEFAULT 19

namespace adit { namespace carplay
{

#if defined(DIPO_DEBUG_MEASURE_VIDEO) || defined(DIPO_DEBUG_OBSERVE_VIDEO)

extern uint64_t GetCurrentTimeNano();
extern uint64_t debug_GetLastScreenProcessNano();

static uint64_t debug_firstTimeStamp = 0;
static bool debug_firstFrame = true;
static pthread_mutex_t debug_mutex = PTHREAD_MUTEX_INITIALIZER;

struct debug_frame
{
    uint64_t process;
    uint64_t stamp;
};

static std::queue<debug_frame> debug_frames;
static std::queue<debug_frame> debug_frames2;

static gboolean debug_videoProbe(GstPad* pad, GstBuffer* buffer, gpointer user)
{
    (void)pad;
    (void)buffer;
    (void)user;

    pthread_mutex_lock(&debug_mutex);
    auto frame = debug_frames.front();
    debug_frames.pop();
    int count = debug_frames.size();
    pthread_mutex_unlock(&debug_mutex);

    uint64_t now = GetCurrentTimeNano();

    // 300ms
    if ((now - frame.process) >= 300 * 1000000)
    {
        printf("PERF %s: time since ScreenStreamProcessData: " \
                "\033[1;31m%llu ms, %d frames in queue\033[0m!\n", __FUNCTION__,
                (now - frame.process) / 1000000, count);
    }
    // if slower than 10 frames or 150ms
    else if (count >= 10 || (now - frame.process) >= 150 * 1000000)
    {
        printf("PERF %s: time since ScreenStreamProcessData: " \
                "\033[1;33m%llu ms, %d frames in queue\033[0m!\n", __FUNCTION__,
                (now - frame.process) / 1000000, count);
    }

#if defined(DIPO_DEBUG_MEASURE_VIDEO)
    guint64 timestamp = GST_BUFFER_TIMESTAMP(buffer) /*+ debug_firstTimeStamp*/;
    printf("PERF %s: time since ScreenStreamProcessData: %llu ns, "\
            "timestamp: %llu, gst: %" GST_TIME_FORMAT ", %d frames in queue\n", __FUNCTION__,
            now - frame.process, timestamp,
            GST_TIME_ARGS(timestamp), count);
#endif
    return TRUE;
}

/*
static gboolean debug_videoEntryProbe(GstPad* pad, GstBuffer* buffer, gpointer user)
{
    (void)pad;
    (void)buffer;
    (void)user;

    pthread_mutex_lock(&debug_mutex);
    auto frame = debug_frames2.front();
    debug_frames2.pop();
    int count = debug_frames2.size();
    pthread_mutex_unlock(&debug_mutex);

    uint64_t now = GetCurrentTimeNano();

    // if slower than 5 frames or 300ms
    if ((now - frame.process) >= 50 * 1000000)
    {
        printf("PERF %s: time since ScreenStreamProcessData: " \
                "\033[1;31m%llu ms, %d packets late\033[0m!\n", __FUNCTION__,
                (now - frame.process) / 1000000, count);
    }
    // if slower than 2 frames or 100ms
    else if ((now - frame.process) >= 10 * 1000000)
    {
        printf("PERF %s: time since ScreenStreamProcessData: " \
                "\033[1;33m%llu ms, %d packets late\033[0m!\n", __FUNCTION__,
                (now - frame.process) / 1000000, count);
    }

    return TRUE;
}
*/
static void debug_attachVideoProbes(GstElement* pipeline)
{
    if (pipeline == nullptr)
        return;

    GstIterator* iter = gst_bin_iterate_elements(GST_BIN(pipeline));
    if (iter == nullptr)
        return;

    gpointer item = nullptr;

    while (GST_ITERATOR_OK == gst_iterator_next(iter, (void**)&item))
    {
        GstElement* element = GST_ELEMENT(item);
        char* name = gst_element_get_name(element);
        if (0 == strncmp(name, "vpudec", 6))
        {
            printf("PERF added video src pad probe to %s\n", name);

            auto pad = gst_element_get_static_pad(element, "src");
            gst_pad_add_buffer_probe(pad, (GCallback)debug_videoProbe, NULL);
            gst_object_unref(pad);
        }
        dipo_safe_call(g_free, name);
    }

    gst_iterator_free(iter);
/*
    iter = gst_bin_iterate_elements(GST_BIN(pipeline));
    if (iter == nullptr)
        return;

    while (GST_ITERATOR_OK == gst_iterator_next(iter, (void**)&item))
    {
        GstElement* element = GST_ELEMENT(item);
        char* name = gst_element_get_name(element);
        if (0 == strncmp(name, "vpudec", 6))
        {
            printf("PERF added video pipeline entry probe to %s\n", name);

            auto pad = gst_element_get_static_pad(element, "sink");
            gst_pad_add_buffer_probe(pad, (GCallback)debug_videoEntryProbe, NULL);
            gst_object_unref(pad);
        }
        dipo_safe_call(g_free, name);
    }

    gst_iterator_free(iter);*/
}
#endif

/* PRQA: Lint Message 826: deactivation because casting mechanism of GObject
 * throws the finding */
/*lint -save -e826*/

static const char* _videoAppSrc = "dipo_video_src";

GstreamerVideoOut::GstreamerVideoOut()
{
    config = nullptr;

    common = new GstreamerCommon;

    appsrc = nullptr;
    videosink = nullptr;
    display = nullptr;
    enoughDataHandlerId = 0;
    videoPlaybackStartHandlerId = 0;
    width = 0;
    height = 0;
    framerate = IVideoOutAdapter::FRAMERATE_30;
    running = false;
    receiver = nullptr;
    sessionId = nullptr;
}

GstreamerVideoOut::~GstreamerVideoOut()
{
    /* PRQA: Lint Message 1506: Stop is not expected to be further overriden */
    /*lint -save -e1506*/
    Stop();
    /*lint -restore*/

    MessageBusUnref();

    delete common;
}

bool GstreamerVideoOut::Initialize(const IConfiguration& inConfig, int inWidth, int inHeight,
        Framerate inFramerate,IVideoReceiver &inReceiver, SessionId inSessionId)
{
    //channelName = "video out";
    receiver = &inReceiver;
    // invalid arguments are better handled at pipeline init
    width = inWidth;
    height = inHeight;
    framerate = inFramerate;

    config = &inConfig;
    sessionId = inSessionId;

    MessageBusAddRef(config->GetNumber("gstreamer-main-loop_thread-prio", GST_MAIN_LOOP_PRIO_DEFAULT));

    return true;
}

bool GstreamerVideoOut::Start()
{
    // store timestamp to give warning for late rendering
    prepareTimeStamp = GetHostTicks();
    // set marker for first frame
    firstFrame = true;
    renderedfirstframe = false;

    if (config == nullptr)
    {
        LOG_ERROR((dipo_gst, "GstreamerVideoOut is not initialized"));
        return false;
    }

    string partialPipeline = config->GetItem("gstreamer-video-pipeline", "");
    LOGD_DEBUG((dipo_gst, "video out pipeline: appsrc ! %s", partialPipeline.c_str()));

    // creates 'pipeline' with 'partial' bin inside
    if (!common->CreatePipeline("dipo_video_out", partialPipeline,
            GstreamerChannelName_VideoOut))
    {
        // error logged
        return false; /* ========== leaving function ========== */
    }

    if (nullptr == (appsrc = gst_element_factory_make("appsrc", _videoAppSrc)))
    {
        LOG_ERROR((dipo_gst, "could not create appsrc %s", _videoAppSrc));
        return false; /* ========== leaving function ========== */
    }

    if (!gst_bin_add(GST_BIN(common->pipeline), appsrc))
    {
        LOG_ERROR((dipo_gst, "could not add appsrc element %s", _videoAppSrc));
        return false; /* ========== leaving function ========== */
    }

    if (!gst_element_link(appsrc, common->partial))
    {
        LOG_ERROR((dipo_gst, "could not link appsrc %s", _videoAppSrc));
        return false; /* ========== leaving function ========== */
    }

    GstCaps* caps = gst_caps_new_simple("video/x-h264",
                                        "width", G_TYPE_INT, width,
                                        "height", G_TYPE_INT, height,
                                        "framerate", GST_TYPE_FRACTION, 0, 1,
                                        "alignment", G_TYPE_STRING, "au",
                                        "stream-format", G_TYPE_STRING, "byte-stream",
                                        NULL);
    if (caps != nullptr)
    {
        gst_app_src_set_caps(GST_APP_SRC(appsrc), caps);
        gst_caps_unref(caps);
    }
    else
    {
        LOG_ERROR((dipo_gst, "could not set video out caps"));
        return false;
    }

    enoughDataHandlerId = g_signal_connect(appsrc, "enough-data",
            G_CALLBACK (&GstreamerVideoOut::enoughDataHandler), this);
    if (enoughDataHandlerId == 0)
        LOG_ERROR((dipo_gst, "could not connect not enough-data video out signal"));

    // Register for "first-videoframe-rendered" signal
    GstIterator* iter = gst_bin_iterate_sinks(GST_BIN(common->partial));
    if (iter == nullptr)
    {
        LOG_ERROR((dipo_gst,"could not get iterator from Gstreamer for the sinks"));
        return false;
    }

#if GST_CHECK_VERSION(1,0,0)
    /* temporary variable */
    GValue gValue = G_VALUE_INIT;
    if(gst_iterator_next(iter, &gValue) == GST_ITERATOR_OK)    //Only one sink will be there
    {
        /* get sink of type GstElement from GValue */
        videosink = (GstElement*)g_value_get_object(&gValue);
#else
    /*get sink element */
    if(gst_iterator_next(iter, (void**)&videosink) == GST_ITERATOR_OK)    //Only one sink will be there
    {
#endif
        string frameRenderedSignalName;

        const std::string name = G_OBJECT_TYPE_NAME(G_OBJECT(videosink));
        if (name.compare("GstApxSink") == 0 )
        {
            wlHandleName = "wl_display";
            frameRenderedSignalName = "first-videoframe-rendered";
        }
        else if((name.compare("GstWaylandSink") == 0 ))
        {
            wlHandleName = "wl-display";
            frameRenderedSignalName = "first-videoframe-rendered";
        }
        else if(name.compare("GstMfxSink") == 0 )
        {
            wlHandleName = "display_ref";
            frameRenderedSignalName = "handoff";
        }
        else
        {
            LOG_ERROR((dipo_gst, "Could not find video sink '%s' to register for first-videoframe-rendered signal", name.c_str()));
            gst_iterator_free(iter);
            gst_object_unref(videosink);
            videosink = nullptr;
            return false;
        }

        LOG_INFO((dipo_gst, "sink used = %s",name.c_str() ));
        //register first frame-buffer rendered signal handler
        videoPlaybackStartHandlerId = g_signal_connect(videosink, frameRenderedSignalName.c_str(),
                    G_CALLBACK (&GstreamerVideoOut::videoPlaybackStartHandler), this);

        if (videoPlaybackStartHandlerId == 0)
        {
            LOG_ERROR((dipo_gst, "could not connect to video playback start signal"));
            gst_object_unref(videosink);
            videosink = nullptr;
            return false;
        }

        gst_iterator_free(iter);
    }
    else
    {
        LOG_ERROR((dipo_gst,"Gstreamer iterator do not return a valid value for sink"));
        gst_iterator_free(iter);
        return false;
    }

    // TODO performance: required?

    uint64_t maxBytes = config->GetNumber("gstreamer-video-max-queue-bytes", 100000);

    g_object_set(G_OBJECT(appsrc),
            "is-live",      TRUE,               // live source
            "do-timestamp", FALSE,              // don't create timestamps
            "block",        FALSE,              // do not block if queue is full
            "min-percent",  0,                  // always empty queue until 0%
            "max-bytes",    (guint64)maxBytes,  // max queue size
            NULL);
    gst_app_src_set_latency(GST_APP_SRC(appsrc), 0, 100000); // TODO performance: configuration? what is appropriate?

#if defined(DIPO_DEBUG_MEASURE_VIDEO) || defined(DIPO_DEBUG_OBSERVE_VIDEO)
    debug_attachVideoProbes(common->partial);
    debug_firstFrame = true;
#endif

    // first set to STATE_READY to capture errors (STATE_READY is synchronous)
    auto ret = gst_element_set_state(GST_ELEMENT(common->pipeline), GST_STATE_READY);
    if (ret == GST_STATE_CHANGE_FAILURE)
    {
        LOG_ERROR((dipo_gst, "gst_element_set_state GST_STATE_READY failed"));
        return false;
    }

    // then set to STATE_PLAYING (is asynchronous)
    ret = gst_element_set_state(GST_ELEMENT(common->pipeline), GST_STATE_PAUSED);
    if (ret == GST_STATE_CHANGE_FAILURE)
    {
        LOG_ERROR((dipo_gst, "gst_element_set_state GST_STATE_PAUSED failed"));
        return false;
    }

    if (!PrepareAndWaitForPushing())
    {
        // error logged
        return false;
    }

    running = true;
    return true;
}

void GstreamerVideoOut::Stop()
{
    bool wasRunning = running;
    running = false;

    /* notify that display becomes invalid */
    uspi::SharedDataSender::instance().notifySurfaceDataExpiration(sessionId);

    if (appsrc != nullptr)
    {
        if (enoughDataHandlerId > 0)
        {
            g_signal_handler_disconnect(appsrc, enoughDataHandlerId);
            enoughDataHandlerId = 0;
        }

        appsrc = nullptr; // is unref'ed by pipeline
    }

    if(videosink != nullptr)
    {
        if (videoPlaybackStartHandlerId > 0)
        {
             g_signal_handler_disconnect(videosink, videoPlaybackStartHandlerId);
             videoPlaybackStartHandlerId = 0;
        }
        gst_object_unref(GST_OBJECT(videosink));
        videosink = nullptr;
    }

    // stop
    common->StopPipeline();

    if (wasRunning)
        LOGD_DEBUG((dipo_gst, "video out stopped"));
}

void GstreamerVideoOut::Push(FrameFragment& inFragment)
{
    if (!running || appsrc == nullptr)
    {
        // DO NOT unset first frame marker as nothing will be rendered
        LOG_WARN_RATE(60, (dipo_gst, "GstreamerVideoOut is not initialized yet (%u times)",
                LOG_RATE_COUNTER));
    }
    else
    {
        // report when first frame entered (diff from Start)
        if (firstFrame)
        {
            // check time difference start until first render
            uint64_t diffNano = GetTickDifference(prepareTimeStamp, GetHostTicks());
            uint64_t diffMilli = diffNano / (1000 * 1000);
            if (diffMilli > 500)
            {
                LOG_WARN((dipo_gst, "video out took more than 500ms (%" PRIu64 "ms) to start rendering!",
                        diffMilli));
            }
            else
                LOGD_DEBUG((dipo_gst, "video out took %" PRIu64 "ms to render", diffMilli));
        }
        // unset first frame marker
        firstFrame = false;

#if defined(DIPO_DEBUG_MEASURE_VIDEO) || defined(DIPO_DEBUG_OBSERVE_VIDEO)
        uint64_t displayTime = inFragment.GetDisplayTime();

        if (debug_firstFrame)
        {
            debug_firstTimeStamp = displayTime;
            debug_firstFrame = false;
        }

        displayTime -= debug_firstTimeStamp;
#endif
       GstBuffer* buffer = gst_buffer_new_and_alloc(inFragment.GetLength());
#if GST_CHECK_VERSION(1,0,0)
        GstMapInfo info;
        if(buffer && gst_buffer_map(buffer, &info, GST_MAP_WRITE))
        {
            if(info.data != NULL)
            {
                memcpy(info.data, inFragment.GetData(), inFragment.GetLength());
                GST_BUFFER_PTS(buffer) = inFragment.GetDisplayTime();
            }
#else
        if(buffer != NULL)
        {
            if(GST_BUFFER_DATA(buffer) != NULL)
            {
                memcpy(GST_BUFFER_DATA(buffer), inFragment.GetData(), inFragment.GetLength());
                GST_BUFFER_TIMESTAMP(buffer) = inFragment.GetDisplayTime();
            }
#endif
            else
            {
                LOG_ERROR((dipo_gst, "gst buffer, data pointer NULL!!, "
                        "Fragment Length:%zd", inFragment.GetLength()));
                gst_buffer_unref(buffer);
                return;
            }
        }
        else
        {
            LOG_ERROR((dipo_gst, "gst buffer NULL!!"));
            return;
        }

#if defined(DIPO_DEBUG_MEASURE_VIDEO) || defined(DIPO_DEBUG_OBSERVE_VIDEO)
        {
            uint64_t now = GetCurrentTimeNano();
            uint64_t process = debug_GetLastScreenProcessNano();
            if (now - process >= 3000000) // if slower than 3ms
            {
                printf("PERF %s: time since ScreenStreamProcessData: " \
                        "\033[1;31m%llu ns\033[0m\n", "GstreamerVideoOut::Push", now - process);
            }
            else if (now - process >= 1000000) // if slower than 1ms
            {
                printf("PERF %s: time since ScreenStreamProcessData: " \
                        "\033[1;33m%llu ns\033[0m\n", "GstreamerVideoOut::Push", now - process);
            }
#if DIPO_DEBUG_MEASURE_VIDEO
            printf("PERF %s: time since ScreenStreamProcessData: %llu ns, "\
                    "timestamp: %llu, gst: %" GST_TIME_FORMAT "\n", "GstreamerVideoOut::Push",
                    now - process, displayTime, GST_TIME_ARGS(displayTime));
#endif
            pthread_mutex_lock(&debug_mutex);
            if (debug_frames.size() == 0 || debug_frames.back().process != process)
                debug_frames.push({ process, now });
            //debug_frames2.push({ process, now });
            pthread_mutex_unlock(&debug_mutex);
        }
#endif // defined(DIPO_DEBUG_MEASURE_VIDEO) || defined(DIPO_DEBUG_OBSERVE_VIDEO)

        int ret = gst_app_src_push_buffer(GST_APP_SRC(appsrc), buffer);
        // takes over buffer, no unref required
        if (ret != 0)
        {
            gst_buffer_unref(buffer); // can't be sure if buffer was unref'ed or not
            // min. interval 60 seconds
            LOG_WARN_RATE(60, (dipo_gst, "gst_app_src_push_buffer failed: %d (%u times)", ret,
                    LOG_RATE_COUNTER));
        }
    }
}

void GstreamerVideoOut::enoughDataHandler(GstElement* inPipeline, GstreamerVideoOut* inMe)
{
    (void)inPipeline;
    (void)inMe;

    // min. interval 60 seconds
    LOG_WARN_RATE(60, (dipo_gst, "got enough-data, render queue is full (%u times)",
            LOG_RATE_COUNTER));
}

void GstreamerVideoOut::videoPlaybackStartHandler(GstElement* inPipeline,GstPad* pad,GstreamerVideoOut* inMe)
{
    (void)inPipeline;
    (void)pad;

    if(inMe != nullptr)
    {
        // make sure to be called only once
        if(!inMe->renderedfirstframe)
        {
            // It is guaranteed that the pipeline is ready to deliver wl display when first frame is rendered
            g_object_get(G_OBJECT (inMe->videosink), inMe->wlHandleName.c_str(), &(inMe->display), NULL);
            if (inMe->display == nullptr)
            {
                LOG_ERROR((dipo_gst, "%s() Couldn't get WL Display", __FUNCTION__));
                inMe->display = nullptr;
            }
            else
            {
                LOG_INFO((dipo_gst, "%s() Got Wl Display = %p from GStreamer", __FUNCTION__, inMe->display));
            }

            uspi::SharedDataSender::instance().transmitSurfaceData(inMe->sessionId, inMe->display);

            //Call the callback registered from videochannel to inform the application
            inMe->receiver->FirstFrameRendered();
            inMe->renderedfirstframe=true;
        }
    }
}

bool GstreamerVideoOut::PrepareAndWaitForPushing()
{
#if 1
    LOGD_DEBUG((dipo_gst, "%" PRIu64 " wait for GST_STATE_PAUSED\n", GetHostTicks() % (GST_SECOND * 10)));
    // wait for gst_element_set_state is necessary
    auto ret = gst_element_get_state(GST_ELEMENT(common->pipeline), NULL, NULL, GST_SECOND);
    if (ret == GST_STATE_CHANGE_FAILURE)
    {
        LOG_ERROR((dipo_gst, "gst_element_get_state GST_STATE_PAUSED failed"));
        return false;
    }
    else if (ret == GST_STATE_CHANGE_ASYNC)
    {
        LOG_WARN((dipo_gst, "gst_element_get_state GST_STATE_PLAYING timed out"));
    }
#endif

    LOGD_DEBUG((dipo_gst, "%" PRIu64 " set to GST_STATE_PLAYING\n", GetHostTicks() % (GST_SECOND * 10)));
    // then set to STATE_PLAYING (is asynchronous)
    ret = gst_element_set_state(GST_ELEMENT(common->pipeline), GST_STATE_PLAYING);
    if (ret == GST_STATE_CHANGE_FAILURE)
    {
        LOG_ERROR((dipo_gst, "gst_element_set_state GST_STATE_PLAYING failed"));
        return false;
    }

    return true;
}

} } // namespace adit { namespace carplay

/*lint -restore*/
